#version 300 es

precision highp float;

uniform float uTime;
uniform vec4 CameraDirection3;

float timeScale = 0.5;
vec4 fxColor = vec4(0,1,1,1);

in vec4 color;
in vec3 mpos;
in vec3 wpos;
in vec3 mnormal;
in vec3 wnormal;

out vec4 outColor;

float heightFactor = 0.0;
vec3 lightPos = vec3(0.0, 1.0, 0.0);
vec3 lightCol = vec3(1.0, 0.0, 0.0);

float lines(float p, float density, float sharpness)
{
	return pow(1.0 - abs(sin(p * density)), sharpness);
}

float grid(vec2 coords, float density, float sharpness)
{
	float a = lines(coords.y * 2.0, density, sharpness);
	float b = lines(coords.x - coords.y, density, sharpness);
	float c = lines(coords.x + coords.y, density, sharpness);
	return pow(max(a, max(b, c)), 3.0);
}

vec4 buildUpFx(float density, float sharpness)
{
	float gridZY = grid(mpos.zy, density, sharpness);
	float gridXZ = grid(mpos.xz, density, sharpness);
	float gridXY = grid(mpos.xy, density, sharpness);
	
	vec3 w = abs(mnormal);
	w = w / (w.x + w.y + w.z);
	
	float triplanarGrid = gridZY * w.x + gridXZ * w.y + gridXY * w.z;
	
	return vec4(triplanarGrid * 0.5);
}

vec3 lighting()
{
	float shininess = 20.0;
	vec3 n = normalize(wnormal);
	vec3 toLight = normalize(lightPos - wpos);
	vec3 viewDir = normalize(CameraDirection3.xyz);
	float diff = max(0.0, dot(n, toLight));
	vec3 h = normalize(viewDir + toLight);
	float spec = diff * pow(max(0.0, dot(n, h)), shininess);
	return (diff + 0.2) * lightCol + spec;
}

float rand3D(in vec3 co){
    return fract(sin(dot(co.xyz ,vec3(12.9898,78.233,144.7272))) * 43758.5453);
}

float simple_interpolate(in float a, in float b, in float x)
{
   return a + smoothstep(0.0,1.0,x) * (b-a);
}

float interpolatedNoise3D(in float x, in float y, in float z)
{
    float integer_x = x - fract(x);
    float fractional_x = x - integer_x;

    float integer_y = y - fract(y);
    float fractional_y = y - integer_y;

    float integer_z = z - fract(z);
    float fractional_z = z - integer_z;

    float v1 = rand3D(vec3(integer_x, integer_y, integer_z));
    float v2 = rand3D(vec3(integer_x+1.0, integer_y, integer_z));
    float v3 = rand3D(vec3(integer_x, integer_y+1.0, integer_z));
    float v4 = rand3D(vec3(integer_x+1.0, integer_y +1.0, integer_z));

    float v5 = rand3D(vec3(integer_x, integer_y, integer_z+1.0));
    float v6 = rand3D(vec3(integer_x+1.0, integer_y, integer_z+1.0));
    float v7 = rand3D(vec3(integer_x, integer_y+1.0, integer_z+1.0));
    float v8 = rand3D(vec3(integer_x+1.0, integer_y +1.0, integer_z+1.0));

    float i1 = simple_interpolate(v1,v5, fractional_z);
    float i2 = simple_interpolate(v2,v6, fractional_z);
    float i3 = simple_interpolate(v3,v7, fractional_z);
    float i4 = simple_interpolate(v4,v8, fractional_z);

    float ii1 = simple_interpolate(i1,i2,fractional_x);
    float ii2 = simple_interpolate(i3,i4,fractional_x);

    return simple_interpolate(ii1 , ii2 , fractional_y);
}

float noise3D(in vec3 coord, in float wavelength)
{
   return interpolatedNoise3D(coord.x/wavelength, coord.y/wavelength, coord.z/wavelength);
}

float layeredNoise(float wavelength, float speed)
{
    vec3 noisepos = wpos;
   	noisepos.xyz += uTime * speed * sign(cos(uTime * timeScale));
   	float noise = abs(noise3D(noisepos, wavelength));
   	noise = smoothstep(0.45 , 0.5, noise) * (1.0 - smoothstep(0.5, 0.55, noise));    
   	return noise;
}

void main() {

	if (wpos.y >= sin(uTime * timeScale)) discard;
    heightFactor = clamp(-0.05 / (sin(uTime * timeScale) - wpos.y - 0.1), 0.0, 1.0);
    float fxIntensity = step(sin(uTime * timeScale) - 0.1, wpos.y);
    float subtleFxIntensity = 0.01 / abs(abs(sin(uTime * timeScale)) - wpos.y - 0.08);
    float solidTransitionGlowIntensity = abs(0.0025 / (sin(uTime * timeScale) - wpos.y - 0.1));
    float solidIntensity = 1.0 - fxIntensity;
    
    float noise = layeredNoise(0.15, 0.14) * 0.5 + layeredNoise(0.14, 0.15) * 0.5;
    noise += layeredNoise(0.1, 0.14) * 0.25 + layeredNoise(0.1, 0.15) * 0.25;
    noise += layeredNoise(0.7, 0.1) * 0.2;
	noise += 0.5;

    vec4 fx = vec4(noise) * fxIntensity + fxIntensity * 0.45 * fxColor;
    fx += noise * fxColor * subtleFxIntensity;
   	   
   	outColor = vec4(lighting(), 1.0) * solidIntensity + fx;
   	outColor += solidTransitionGlowIntensity;   
}